<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class InvestmentPlan extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'description',
        'min_amount',
        'max_amount',
        'interest_rate',
        'duration',
        'interest_type',
        'status',
    ];

    public function investments()
    {
        return $this->hasMany(Investment::class);
    }

    public function getDurationInDaysAttribute(): int
    {
        return match (strtolower($this->interest_type)) {
            'daily'   => $this->duration,
            'weekly'  => $this->duration * 7,
            'monthly' => $this->duration * 30,
            'yearly'  => $this->duration * 365,
            'end'     => 0,
            default   => 0,
        };
    }

    /**
     * (Optional) Get the actual end date based on start_date + duration.
     */
    public function getExpectedEndDateAttribute()
    {
        $start = Carbon::parse($this->start_date);

        return match (strtolower($this->interest_type)) {
            'daily'   => $start->addDays($this->duration),
            'weekly'  => $start->addWeeks($this->duration),
            'monthly' => $start->addMonths($this->duration),
            'yearly'  => $start->addYears($this->duration),
            'end'     => $start,
            default   => $start,
        };
    }


    public function getTotalReturnPercentageAttribute()
    {
        // Total interest % across full duration
        return $this->interest_rate * $this->duration;
    }

    public function getMinAmountPercentageAttribute()
    {
        if ($this->max_amount <= 0) return 0;
        return round(($this->min_amount / $this->max_amount) * 100, 2);
    }

    public function getMaxAmountPercentageAttribute()
    {
        return 100;
    }
}
