<?php

namespace App\Http\Controllers;

use App\Models\Withdrawal;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class WithdrawalController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function userwithdrawal(Request $request)
    {
        $pageTitle = 'Withdrawal';

        return view('user/withdrawal', compact('pageTitle'));
    }

    public function userwithdrawalaccess(Request $request)
    {
        $pageTitle = 'Withdrawal Access';

        return view('user/withdrawal_access', compact('pageTitle'));
    }


    public function verifyAccess(Request $request)
    {
        $request->validate([
            'password' => 'required'
        ]);

        if (Hash::check($request->password, Auth::user()->password)) {

            session([
                'withdrawal_verified' => true,
                'withdrawal_verified_time' => now()
            ]);

            return redirect()->route('user.withdrawal');
        }

        return back()->withErrors(['password' => 'Incorrect password']);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function userwithdrawalstore(Request $request)
    {
        $request->validate([
            'method_type' => 'required|in:bank,crypto',
            'amount' => 'required|numeric|min:10|regex:/^[^<>]*$/',
        ]);

        $user = Auth::user();

        if ($user->balance < $request->amount) {
            return back()->with('error', 'Insufficient balance.');
        }

        DB::beginTransaction();
        try {
            $data = [
                'user_id' => $user->id,
                'method_type' => $request->method_type,
                'amount' => $request->amount,
                'status' => 'pending',
            ];

            if ($request->method_type === 'bank') {
                $request->validate([
                    'bank_name' => 'required|string',
                    'account_name' => 'required|string',
                    'account_number' => 'required|string',
                ]);
                $data = array_merge($data, $request->only('bank_name', 'account_name', 'account_number'));
            }

            if ($request->method_type === 'crypto') {
                $request->validate([
                    'crypto_type' => 'required|string',
                    'wallet_address' => 'required|string',
                    'network' => 'required|string',
                ]);
                $data = array_merge($data, $request->only('crypto_type', 'wallet_address', 'network'));
            }

            // Deduct user balance
            $user->balance -= $request->amount;
            $user->save();

            // Save withdrawal
            Withdrawal::create($data);

            DB::commit();

            return back()->with('success', 'Withdrawal request submitted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Something went wrong. Try again.');
        }
    }


    //ADMIN
    public function adminwithdrawindex(Request $request)
    {
        $data = Withdrawal::latest()->paginate(25);

        $pageTitle = 'User Withdrawals';

        return view('admin/withdraw/index', compact('pageTitle', 'data'));
    }

    public function adminwithdrawdetails(Request $request, $id)
    {
        $data = Withdrawal::findorfail($id);

        $pageTitle = 'User Withdrawals';

        return view('admin/withdraw/details', compact('pageTitle', 'data'));
    }


    public function adminapprovewith($id)
    {
        $withdrawal = Withdrawal::findOrFail($id);

        if ($withdrawal->status !== 'pending') {
            return back()->with('error', 'Withdrawal already processed.');
        }

        DB::beginTransaction();

        try {
            $withdrawal->update([
                'status' => 'approved',
                'approved_by' => Auth::id(),
                'approved_at' => now(),
            ]);

            DB::commit();

            return back()->with('success', 'Withdrawal approved successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Something went wrong.');
        }
    }

    public function adminrejectwith($id)
    {
        $withdrawal = Withdrawal::findOrFail($id);

        if ($withdrawal->status !== 'pending') {
            return back()->with('error', 'Withdrawal already processed.');
        }

        DB::beginTransaction();

        try {
            $user = $withdrawal->user;

            // Refund amount to user balance
            if ($user) {
                $user->balance += $withdrawal->amount;
                $user->save();
            }

            $withdrawal->update([
                'status' => 'rejected',
                'approved_by' => Auth::id(),
                'approved_at' => now(),
            ]);

            DB::commit();

            return back()->with('success', 'Withdrawal rejected and amount refunded.');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Something went wrong.');
        }
    }
}
