<?php

namespace App\Http\Controllers;

use App\Models\TradeExpert;
use Illuminate\Support\Facades\DB;
use App\Models\UserCopyTrade;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use App\Models\Notification;
use App\Mail\UserCopyTradeStartedMail;
use App\Mail\AdminCopyTradeStartedMail;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use App\Mail\UserCopyTradeStoppedMail;
use App\Mail\AdminCopyTradeStoppedMail;



class UserCopyTradeController extends Controller
{
    public function index()
    {
        $copyTrades = UserCopyTrade::with('expert')
            ->where('user_id', Auth::id())
            ->get();

        return view('user.copytrade.index', compact('copyTrades'));
    }

    public function startusercopy(Request $request, $expert_id)
    {
        $expert = TradeExpert::findOrFail($expert_id);
        $user = Auth::user();

        $request->validate([
            'amount' => ['required', 'numeric', 'min:' . $expert->min_amount],
        ]);

        if ($user->balance < $request->amount) {
            return back()->with('error', 'Insufficient balance to start copy trade.');
        }

        $amount = (float) $request->amount;
        $scalingFactor = $amount / max(1, $expert->equity);

        $copyTrade = UserCopyTrade::create([
            'user_id' => $user->id,
            'trade_expert_id' => $expert->id,
            'amount' => $amount,
            'current_value' => $amount,
            'status' => 'active',
            'started_at' => now(),
            'scaling_factor' => $scalingFactor,
        ]);

        $user->decrement('balance', $amount);
        $expert->increment('followers');

        /*
    |--------------------------------------------------------------------------
    | EMAILS & NOTIFICATIONS
    |--------------------------------------------------------------------------
    */
        $admin = User::where('role', 'admin')->first();

        try {

            // USER EMAIL
            Mail::to($user->email)
                ->queue(new UserCopyTradeStartedMail($user, $expert, $copyTrade));

            // ADMIN EMAIL
            if ($admin) {
                Mail::to($admin->email)
                    ->queue(new AdminCopyTradeStartedMail($user, $expert, $copyTrade));
            }

            // USER NOTIFICATION
            Notification::create([
                'id' => Str::uuid(),
                'notifiable_type' => User::class,
                'notifiable_id' => $user->id,
                'type' => 'Copy Trade Started',
                'data' => [
                    'message' => "You started copying {$expert->name}.",
                    'amount' => $amount,
                    'status' => 'Active',
                ],
            ]);

            // ADMIN NOTIFICATION
            if ($admin) {
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $admin->id,
                    'type' => 'New Copy Trade',
                    'data' => [
                        'message' => "{$user->firstname} started a copy trade.",
                        'expert' => $expert->name,
                        'amount' => $amount,
                    ],
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Copy trade notify error: ' . $e->getMessage());
        }

        return redirect()->route('user.copytrade')
            ->with('success', "You are now copying {$expert->name}!");
    }


    public function userstopcopy($id)
    {
        $copyTrade = UserCopyTrade::where('user_id', Auth::id())
            ->where('id', $id)
            ->firstOrFail();

        $user = $copyTrade->user;
        $finalAmount = $copyTrade->current_value;

        DB::transaction(function () use ($copyTrade, $user, $finalAmount) {

            $user->increment('profit', $finalAmount);

            $copyTrade->update([
                'status' => 'stopped',
                'stopped_at' => Carbon::now(),
            ]);

            DB::table('copy_trade_logs')->insert([
                'user_id' => $user->id,
                'trade_expert_id' => $copyTrade->trade_expert_id,
                'profit_amount' => $finalAmount - $copyTrade->amount,
                'percent_change' => ($finalAmount - $copyTrade->amount) / $copyTrade->amount * 100,
                'note' => "Trade stopped and credited to user",
                'created_at' => now(),
            ]);
        });

        /*
    |--------------------------------------------------------------------------
    | EMAILS & NOTIFICATIONS
    |--------------------------------------------------------------------------
    */
        $admin = User::where('role', 'admin')->first();

        try {

            // USER EMAIL
            Mail::to($user->email)
                ->queue(new UserCopyTradeStoppedMail($user, $copyTrade, $finalAmount));

            // ADMIN EMAIL
            if ($admin) {
                Mail::to($admin->email)
                    ->queue(new AdminCopyTradeStoppedMail($user, $copyTrade, $finalAmount));
            }

            // USER NOTIFICATION
            Notification::create([
                'id' => Str::uuid(),
                'notifiable_type' => User::class,
                'notifiable_id' => $user->id,
                'type' => 'Copy Trade Stopped',
                'data' => [
                    'message' => 'Your copy trade has been stopped.',
                    'amount' => $finalAmount,
                    'status' => 'Stopped'
                ],
            ]);

            // ADMIN NOTIFICATION
            if ($admin) {
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $admin->id,
                    'type' => 'User Stopped Copy Trade',
                    'data' => [
                        'message' => "{$user->firstname} stopped a copy trade.",
                        'amount' => $finalAmount,
                    ],
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Copy trade stop notify error: ' . $e->getMessage());
        }

        return back()->with('success', 'Copy trade stopped and credited successfully.');
    }
}
