<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Staking;
use App\Models\StakingPlan;
use Illuminate\Support\Str;
use App\Models\Notification;
use Illuminate\Http\Request;
use App\Mail\AdminNewStakingMail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Mail\UserStakingStartedMail;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;

class StakingController extends Controller
{
    public function showuserstaking(Request $request)
    {
        $stake = StakingPlan::where('status', 'active')->get();

        $plan = Staking::where('user_id', auth()->id())->latest()->paginate(25);

        $totalplan = Staking::where('user_id', auth()->id())->count();

        $activeplan = Staking::where('user_id', auth()->id())
            ->where('status', 'active')->count();

        $closedplan = Staking::where('user_id', auth()->id())
            ->where('status', 'completed')->count();

        $userId = auth()->id();

        $totalStaked = Staking::where('user_id', $userId)
            ->sum('amount');

        $totalActiveStaked = Staking::where('user_id', $userId)
            ->where('status', 'active')
            ->sum('amount');

        $totalReturn = Staking::where('user_id', $userId)
            ->sum(DB::raw('amount + interest_earned'));


        return view('user.staking', compact('stake', 'plan', 'activeplan', 'totalplan', 'closedplan', 'totalStaked', 'totalActiveStaked', 'totalReturn'));
    }

    public function userstorestaking(Request $request)
    {
        // dd($request->all());

        $plan = StakingPlan::findOrFail($request->staking_plan_id);

        $amount = $request->amount;
        $days   = (int) $request->duration;
        $rate   = $plan->interest_rate;

        if ($amount < $plan->minimum_amount || $amount > $plan->maximum_amount) {
            return back()->withErrors('Invalid amount');
        }

        $user = Auth::user();

        if ($user->balance < $request->amount) {
            return back()->with('error', 'Insufficient wallet balance.');
        }

        $interest = ($amount * $rate / 100) * $days;

        $endDate  = now()->addDays($days);

        $staking = Staking::create([
            'user_id' => $user->id,
            'staking_plan_id' => $plan->id,
            'cryptocurrencies_id' => $plan->cryptocurrency_id,
            'amount' => $amount,
            'duration' => $days,
            'interest_rate' => $rate,
            'interest_earned' => $interest,
            'status' => 'active',
            'start_date' => now(),
            'end_date' => $endDate,
        ]);

        $user->decrement('balance', $amount);

        /*
        |--------------------------------------------------------------------------
        | EMAILS & NOTIFICATIONS
        |--------------------------------------------------------------------------
        */
        $admin = User::where('role', 'admin')->first();

        try {

            // USER MAIL
            Mail::to($user->email)
                ->send(new UserStakingStartedMail($user, $plan, $staking));

            // ADMIN MAIL
            if ($admin) {
                Mail::to($admin->email)
                    ->send(new AdminNewStakingMail($user, $plan, $staking));
            }

            // ADMIN NOTIFICATION
            if ($admin) {
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $admin->id,
                    'type' => 'New Staking Created',
                    'data' => [
                        'message' => "{$user->firstname} {$user->lastname} started a staking plan",
                        'plan' => $plan->name,
                        'amount' => $amount,
                        'duration' => $days . ' days',
                        'interest_rate' => $rate . '%',
                        'status' => 'Active',
                    ],
                ]);
            }

            // USER NOTIFICATION
            Notification::create([
                'id' => Str::uuid(),
                'notifiable_type' => User::class,
                'notifiable_id' => $user->id,
                'type' => 'Staking Started',
                'data' => [
                    'message' => "Your staking plan {$plan->name} has started successfully.",
                    'plan' => $plan->name,
                    'amount' => $amount,
                    'duration' => $days . ' days',
                    'expected_interest' => $interest,
                    'end_date' => $endDate->format('Y-m-d'),
                    'status' => 'Active',
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Staking Mail Error: ' . $e->getMessage());
        }

        return back()->with('success', 'Stake placed successfully');
    }


    //ADMIN
    public function adminuserstake(Request $request)
    {
        $pageTitle = 'Users Stakes';

        $trade = Staking::latest()->paginate(30);

        return view('admin.staking.trades', compact('trade', 'pageTitle'));
    }
}
