<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Signal;
use Illuminate\Support\Str;
use App\Models\Notification;
use Illuminate\Http\Request;
use App\Models\PaymentGateway;
use App\Models\SignalPurchase;
use App\Mail\SignalApprovedMail;
use App\Mail\SignalRejectedMail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use App\Mail\SignalPurchasedUserMail;
use App\Mail\SignalPurchasedAdminMail;

class SignalController extends Controller
{
    public function showusersignal(Request $request)
    {
        $pageTitle = 'Purchase Signals';

        $total = Signal::count();

        $purchases = SignalPurchase::where('user_id', auth()->id())->get();

        $signals = Signal::where('is_active', true)->latest()->get();

        return view('user.signals', compact('pageTitle', 'total', 'signals', 'purchases'));
    }

    public function showusersignaldetails(Request $request, $id)
    {
        $signal = Signal::findorfail($id);

        $pageTitle = 'Buy ' . $signal->title;

        $pay = PaymentGateway::where('status', 'active')->get();

        $amount = $signal->price;

        return view('user.signal_details', compact('pageTitle', 'signal', 'pay', 'amount'));
    }


    public function userstoresignal(Request $request, Signal $signal)
    {
        $validated = $request->validate([
            'pay_image' => 'required|image|mimes:jpeg,png,jpg|max:5120',
        ]);

        $user = auth()->user();
        $admin = User::where('role', 'admin')->first();

        $already = SignalPurchase::where([
            'user_id' => $user->id,
            'signal_id' => $signal->id
        ])->exists();

        if ($already) return back()->with('info', 'Already purchased');

        $payImagePath = null;

        if ($request->hasFile('pay_image')) {

            $file = $request->file('pay_image');
            $imageName = Str::random(12) . '.' . $file->getClientOriginalExtension();

            $uploadPath = public_path('uploads/signal');
            if (!file_exists($uploadPath)) {
                mkdir($uploadPath, 0755, true);
            }

            $file->move($uploadPath, $imageName);

            $payImagePath = 'uploads/signal/' . $imageName;
        }

        $signal = SignalPurchase::create([
            'user_id' => $user->id,
            'signal_id' => $request->signal_id,
            'payment_gateway_id' => $request->payment_gateway_id,
            'amount' => $request->price,
            'pay_image' => $payImagePath,
            'status' => 'pending'
        ]);

        try {

            // User email
            Mail::to($user->email)
                ->send(new SignalPurchasedUserMail($user, $signal));

            // Admin email
            if ($admin) {
                Mail::to($admin->email)
                    ->send(new SignalPurchasedAdminMail($user, $signal));
            }

            // 🔹 Notify Admin
            if ($admin) {
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $admin->id,
                    'type' => 'New Signal Purchase',
                    'data' => [
                        'message'   => "{$user->firstname} {$user->lastname} subscribed to {$signal->signal->title} signal",
                        'signal_id' => $signal->id,
                        'amount'    => $signal->price,
                        'status'    => 'Pending',
                    ],
                ]);
            }

            // 🔹 Notify User
            Notification::create([
                'id' => Str::uuid(),
                'notifiable_type' => User::class,
                'notifiable_id' => $user->id,
                'type' => 'Signal Subscription',
                'data' => [
                    'message'   => "You have successfully subscribed to {$signal->signal->title} signal. Awaiting approval.",
                    'signal_id' => $signal->id,
                    'amount'    => $signal->price,
                    'status'    => 'Pending',
                ],
            ]);
        } catch (\Exception $e) {
        }

        return redirect()
            ->route('user.signals')
            ->with('success', 'Signal unlocked successfully!');
    }



    //ADMIN SIGNAL

    public function adminsignalindex(Request $request)
    {
        $pageTitle = 'Signals';

        $signal = Signal::all();

        return view('admin.signal.index', compact('pageTitle', 'signal'));
    }

    public function adminusersignal(Request $request)
    {
        $pageTitle = 'Signals Purchases';

        $signal = SignalPurchase::paginate(30);

        return view('admin.signal.deposit', compact('pageTitle', 'signal'));
    }


    public function adminsignaldeposit(Request $request, $id)
    {
        $signal = SignalPurchase::findorfail($id);

        $pageTitle = 'Signal Deposits Details';

        return view('admin.signal.details', compact('pageTitle', 'signal'));
    }

    public function adminsignaldepositupdate(Request $request, $id)
    {
        $validated = $request->validate([
            'status' => 'required|in:approved,rejected',
        ]);

        $signal = SignalPurchase::findOrFail($id);
        $newStatus = $validated['status'];
        $user = $signal->user;

        $signal->status = $newStatus;
        $signal->save();

        /*
    |--------------------------------------------------------------------------
    | EMAILS
    |--------------------------------------------------------------------------
    */
        try {
            if ($newStatus === 'approved') {
                Mail::to($user->email)->queue(new SignalApprovedMail($signal));
            } elseif ($newStatus === 'rejected') {
                Mail::to($user->email)->queue(new SignalRejectedMail($signal));
            }
        } catch (\Exception $e) {
            Log::error('Signal status mail failed: ' . $e->getMessage());
        }

        /*
    |--------------------------------------------------------------------------
    | NOTIFICATIONS
    |--------------------------------------------------------------------------
    */
        try {
            $admin = User::where('role', 'admin')->first();

            // USER NOTIFICATION
            $userMessage = match ($newStatus) {
                'approved' => "Your signal purchase of $ {$signal->amount} has been approved.",
                'rejected' => "Your signal purchase of $ {$signal->amount} was rejected.",
                default => "Your signal purchase status changed to {$newStatus}.",
            };

            Notification::create([
                'id' => Str::uuid(),
                'notifiable_type' => User::class,
                'notifiable_id' => $user->id,
                'type' => "Signal {$newStatus}",
                'data' => [
                    'message' => $userMessage,
                    'signal_id' => $signal->id,
                    'status' => $newStatus,
                ],
            ]);

            // ADMIN NOTIFICATION
            if ($admin) {
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $admin->id,
                    'type' => "Signal {$newStatus} Update",
                    'data' => [
                        'message' => "You updated {$user->firstname} {$user->lastname}'s signal purchase ($ {$signal->amount}) to {$newStatus}.",
                        'signal_id' => $signal->id,
                        'status' => $newStatus,
                    ],
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Signal notification creation failed: ' . $e->getMessage());
        }

        return redirect()->route('admin.signal.deposit')
            ->with('success', 'Signal purchase status updated successfully.');
    }


    public function adminsignaladd(Request $request)
    {
        $pageTitle = 'Add Signals';

        return view('admin.signal.add', compact('pageTitle'));
    }


    public function admincreatesignal(Request $request)
    {
        Signal::create($request->all());

        return redirect()->route('admin.signal.index')->with('success', 'Signal created');
    }


    public function adminsignaledit(Request $request, $id)
    {
        $signal = Signal::findorfail($id);

        $pageTitle = 'Manage Signals';

        return view('admin.signal.edit', compact('pageTitle', 'signal'));
    }


    public function adminupdatesignal(Request $request, $id)
    {
        $signal = Signal::findOrFail($id);

        $signal->update($request->only([
            'title',
            'description',
            'pair',
            'direction',
            'entry_price',
            'take_profit',
            'stop_loss',
            'price',
            'status',
            'strength',
            'is_active',
            'expires_at'
        ]));

        return redirect()
            ->route('admin.signal.index')
            ->with('success', 'Signal updated successfully');
    }


    public function signaldestroy(Signal $signal)
    {
        $signal->delete();
        return redirect()->route('admin.signal.index')->with('success', 'Signal deleted successfully.');
    }
}
