<?php

namespace App\Http\Controllers;

use App\Models\LiveTrade;
use App\Mail\LiveTradePlacedMail;
use App\Mail\AdminLiveTradeAlertMail;
use App\Models\WebsiteSetting;
use Illuminate\Support\Facades\Mail;
use App\Models\User;
use Illuminate\Support\Str;
use App\Models\Notification;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Http\Request;

class LiveTradeController extends Controller
{

    public function userlivestore(Request $request)
    {
        $request->validate([
            'symbol' => 'required|string',
            'side' => 'required|in:buy,sell',
            'amount' => 'required|numeric|min:1',
            'leverage' => 'required|numeric|min:1',
            'order_type' => 'required|string',
            'expiration' => 'required|string',
        ]);

        $user = auth()->user();

        // 🔒 Check if user has enough balance
        if ($user->balance < $request->amount) {
            return response()->json(['error' => 'Insufficient balance'], 400);
        }

        $symbol = strtoupper($request->symbol);
        $currentPrice = null;

        // ✅ Detect if crypto (Binance) or stock/forex (TwelveData)
        if (str_ends_with($symbol, 'USDT')) {
            $priceData = Http::get("https://api.binance.com/api/v3/ticker/price", [
                'symbol' => $symbol
            ])->json();
            $currentPrice = $priceData['price'] ?? null;
        } else {
            $response = Http::get("https://api.twelvedata.com/quote", [
                'symbol' => $symbol,
                'apikey' => config('services.twelvedata.key')
            ])->json();

            $currentPrice = $response['close'] ?? null;
        }

        if (!$currentPrice || !is_numeric($currentPrice)) {
            Log::error('Price fetch failed for symbol ' . $symbol, ['response' => $priceData ?? $response ?? []]);
            return response()->json(['error' => 'Invalid symbol or market data unavailable'], 400);
        }

        $units = $request->amount / $currentPrice;

        // ⚙️ Start database transaction for safety
        DB::beginTransaction();

        try {
            // 💰 Deduct user balance
            $user->balance -= $request->amount;
            $user->save();

            // 📈 Create trade record
            $trade = LiveTrade::create([
                'user_id' => $user->id,
                'symbol' => $symbol,
                'asset_name' => $request->asset_name,
                'market_type' => $request->market_type,
                'side' => $request->side,
                'entry_price' => $currentPrice,
                'amount' => $request->amount,
                'units' => $units,
                'leverage' => $request->leverage,
                'expiration' => $request->expiration,
                'order_type' => $request->order_type,
                'expires_at' => $request->expiration ? now()->addMinutes($this->parseExpiration($request->expiration)) : null,
                'current_value' => $request->amount,
                'profit_loss' => 0,
            ]);

            DB::commit();


            /* ===============================
           EMAIL NOTIFICATIONS
        ================================ */
            try {
                // ✉️ Email user
                Mail::to($user->email)->queue(new LiveTradePlacedMail($trade));

                // ✉️ Email admin
                $adminEmail = WebsiteSetting::value('site_email');
                if ($adminEmail) {
                    Mail::to($adminEmail)->queue(new AdminLiveTradeAlertMail($trade));
                }
            } catch (\Exception $e) {
                Log::error("LiveTrade email failed: " . $e->getMessage());
            }

            /* ===============================
           DATABASE NOTIFICATIONS
        ================================ */
            try {
                $admin = User::where('role', 'admin')->first();

                // 🔔 User Notification
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $user->id,
                    'type' => 'Live Trade Started',
                    'data' => json_encode([
                        'message' => "Your live trade on {$symbol} ({$request->side}) has been placed successfully.",
                        'symbol' => $symbol,
                        'amount' => $request->amount,
                    ]),
                ]);

                // 🔔 Admin Notification
                if ($admin) {
                    Notification::create([
                        'id' => Str::uuid(),
                        'notifiable_type' => User::class,
                        'notifiable_id' => $admin->id,
                        'type' => 'New Live Trade',
                        'data' => json_encode([
                            'message' => "{$user->firstname} {$user->lastname} placed a live trade on {$symbol}.",
                            'amount' => $request->amount,
                            'symbol' => $symbol,
                        ]),
                    ]);
                }
            } catch (\Exception $e) {
                Log::error("LiveTrade notification failed: " . $e->getMessage());
            }

            /* ===============================
           USER ACTIVITY LOG
        ================================ */
            logUserActivity('Live Trade', "Placed a {$request->side} trade on {$symbol} worth \${$request->amount}");

            return response()->json([
                'message' => 'Trade placed successfully',
                'trade' => $trade,
                'new_balance' => $user->balance,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Trade placement failed: ' . $e->getMessage());
            return response()->json(['error' => 'Something went wrong while placing the trade'], 500);
        }
    }

    private function parseExpiration($input)
    {
        $map = [
            'm' => 1,     // minutes
            'h' => 60,    // hours
            'd' => 1440,  // days
        ];

        preg_match('/(\d+)([mhd])/', $input, $matches);
        if (!$matches) return null;

        $value = (int)$matches[1];
        $unit = $matches[2];

        return $value * $map[$unit];
    }


    public function userstopTrade($id)
    {
        $trade = LiveTrade::findOrFail($id);

        // ✅ Ensure trade is still open
        if ($trade->status !== 'open') {
            return response()->json([
                'message' => 'Trade already closed or completed.',
            ], 400);
        }

        // ✅ Start database transaction to ensure atomic updates
        DB::beginTransaction();

        try {
            $user = $trade->user;

            // 💰 Credit user with current trade value (not exceeding balance safety)
            if ($user && $trade->current_value > 0) {
                $user->balance += $trade->current_value;
                $user->save();
            }

            // ✅ Mark trade as closed
            $trade->update([
                'status' => 'closed',
                'expires_at' => now(),
            ]);

            DB::commit();

            return response()->json([
                'message' => '✅ Trade stopped successfully.',
                'balance' => $user->balance,
                'trade' => $trade,
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            Log::error("❌ Failed to stop trade #{$trade->id}: " . $e->getMessage());

            return response()->json([
                'message' => 'Something went wrong while stopping the trade.',
            ], 500);
        }
    }
}
