<?php

namespace App\Http\Controllers;

use App\Models\Investment;
use Illuminate\Support\Facades\Mail;
use App\Mail\AdminNewInvestmentMail;
use App\Mail\InvestmentStartedMail;
use App\Models\WebsiteSetting;
use Illuminate\Support\Str;
use App\Models\User;
use App\Models\Notification;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use App\Models\InvestmentPlan;
use App\Models\LiveTrade;
use Carbon\Carbon;
use Illuminate\Http\Request;

class InvestmentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function storeuserinvestment(Request $request, $plan_id)
    {
        $plan = InvestmentPlan::findOrFail($plan_id);
        $user = Auth::user();

        $minAmount = (float) $plan->min_amount;
        $maxAmount = (float) $plan->max_amount;
        $interestRate = (float) $plan->interest_rate;
        $duration = (int) $plan->duration;
        $interestType = strtolower($plan->interest_type);

        // ✅ Validation
        $request->validate([
            'amount' => ['required', 'numeric', 'min:' . $minAmount, 'max:' . $maxAmount],
        ]);

        $amount = (float) $request->amount;

        // 🚫 Check user balance
        if ($user->balance < $amount) {
            return back()->with('error', 'Insufficient balance to invest.');
        }

        // 🧮 Determine duration in days
        switch ($interestType) {
            case 'daily':
                $durationDays = $duration;
                break;
            case 'weekly':
                $durationDays = $duration * 7;
                break;
            case 'monthly':
                $durationDays = $duration * 30;
                break;
            case 'yearly':
                $durationDays = $duration * 365;
                break;
            default:
                $durationDays = $duration;
        }

        // 🧮 Calculate total expected profit (for info only)
        $totalProfit = ($amount * $interestRate / 100) * $duration;
        $expectedReturn = $amount + $totalProfit;

        // 🕒 Dates
        $startDate = now();
        $endDate = $startDate->copy()->addDays($durationDays);

        // 💾 Save Investment (profit starts at zero)
        $investment = Investment::create([
            'user_id' => $user->id,
            'investment_plan_id' => $plan->id,
            'amount' => $amount,
            'profit' => 0, // ✅ Start at 0
            'expected_return' => $expectedReturn,
            'duration' => $duration,
            'start_date' => $startDate,
            'end_date' => $endDate,
            'days_completed' => 0,
            'status' => 'active',
        ]);

        // 💰 Deduct from user balance
        $user->decrement('balance', $amount);

        // Activity Log
        logUserActivity('Investment', "Invested $$amount into {$plan->name} plan");

        // ADMIN email
        $adminEmail = WebsiteSetting::value('site_email');

        try {
            // Email user
            Mail::to($user->email)->queue(new InvestmentStartedMail($investment));

            // Email admin
            if ($adminEmail) {
                Mail::to($adminEmail)->queue(new AdminNewInvestmentMail($investment));
            }
        } catch (\Exception $e) {
            Log::error("Investment email failed: " . $e->getMessage());
        }

        // Database Notifications
        try {
            // Notify User
            Notification::create([
                'id' => Str::uuid(),
                'notifiable_type' => User::class,
                'notifiable_id' => $user->id,
                'type' => 'Investment Started',
                'data' => json_encode([
                    'message' => "Your investment of $$amount in {$plan->name} has started.",
                    'plan' => $plan->name,
                    'amount' => $amount,
                    'expected_return' => $expectedReturn,
                ]),
            ]);

            // Notify Admin
            $admin = User::where('role', 'admin')->first();
            if ($admin) {
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $admin->id,
                    'type' => 'New Investment',
                    'data' => json_encode([
                        'message' => "{$user->firstname} {$user->lastname} invested $$amount in {$plan->name}",
                        'user' => $user->email,
                        'amount' => $amount,
                    ]),
                ]);
            }
        } catch (\Exception $e) {
            Log::error("Investment notification failed: " . $e->getMessage());
        }


        return redirect()->route('user.portfolio')
            ->with('success', 'Investment successful! Expected return: $' . number_format($expectedReturn, 2));
    }

    public function adminuserinvest(Request $request)
    {
        $pageTitle = 'Users Investments';

        $invest = Investment::latest()->paginate(25);

        return view('admin/investment/invest', compact('invest', 'pageTitle'));
    }

    public function adminuserlivetrade(Request $request)
    {
        $pageTitle = 'Users Live Trades';

        $invest = LiveTrade::latest()->paginate(25);

        return view('admin/live/trade', compact('invest', 'pageTitle'));
    }
}
