<?php

namespace App\Http\Controllers;

use App\Models\Deposit;
use App\Models\User;
use App\Models\Notification;
use App\Mail\DepositRejectedMail;
use App\Mail\DepositApprovedMail;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use App\Mail\NewDepositAlertMail;
use App\Mail\DepositPendingMail;
use App\Models\WebsiteSetting;
use Intervention\Image\Laravel\Facades\Image;
use Intervention\Image\Encoders\JpegEncoder;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use App\Models\PaymentGateway;
use Illuminate\Http\Request;

class DepositController extends Controller
{
    //USER SECTION
    public function userdeposit(Request $request)
    {
        $pageTitle = 'Deposit';

        $paymentGateways = PaymentGateway::where('status', 'active')->get();

        return view('user/deposit', compact('pageTitle', 'paymentGateways'));
    }

    // Step 2: Handle POST from create form and redirect to confirm page
    public function handleConfirm(Request $request)
    {
        $validated = $request->validate([
            'payment_gateway_id' => 'required|exists:payment_gateways,id',
            'amount' => 'required|numeric|min:1',
        ]);


        return redirect()->route('user.deposit_confirm', [
            'payment_gateway_id' => $validated['payment_gateway_id'],
            'amount' => $validated['amount'],
        ]);
    }


    public function confirmDeposit(Request $request)
    {
        $gateway = PaymentGateway::findOrFail($request->payment_gateway_id);
        $amount = $request->amount;

        return view('user.deposit_confirm', compact('gateway', 'amount'));
    }


    public function storeDeposit(Request $request)
    {
        $validated = $request->validate([
            'payment_gateway_id' => 'required|exists:payment_gateways,id',
            'amount' => 'required|numeric|min:1|regex:/^[^<>]*$/|',
            'pay_image' => 'required|image|mimes:jpeg,png,jpg|max:5120', // 5MB limit
            'remark' => 'nullable|string|regex:/^[^<>]*$/|',
        ]);

        $filename = null;

        if ($request->hasFile('pay_image')) {
            $file = $request->file('pay_image');

            // ✅ Generate random filename and always use .jpg
            $randomStr = Str::random(15);
            $filename = $randomStr . '.jpg';

            // ✅ Define destination path
            $destinationPath = public_path('upload/deposit/');

            // ✅ Create folder if missing
            if (!file_exists($destinationPath)) {
                mkdir($destinationPath, 0755, true);
            }

            // ✅ Decode, sanitize, and re-encode to JPG (strips malicious code)
            $image = Image::read($file->getRealPath())
                ->orient()
                ->encode(new JpegEncoder(quality: 85));

            // ✅ Save safely
            $image->save($destinationPath . $filename);
        }

        // ✅ Store deposit record
        $deposit = Deposit::create([
            'user_id' => Auth::id(),
            'payment_gateway_id' => $validated['payment_gateway_id'],
            'amount' => $validated['amount'],
            'pay_image' => $filename,
            'remark' => $validated['remark'] ?? null,
            'status' => 'pending',
        ]);

        $user = $deposit->user;
        $admin = User::where('role', 'admin')->first();

        // ✅ Activity Log (user action)
        if (function_exists('logUserActivity')) {
            logUserActivity('Deposit', 'You made a deposit of $' . number_format($deposit->amount, 2));
        } else {
            Log::warning('⚠️ logUserActivity() not found. Ensure helper is globally autoloaded.');
        }

        // ✅ Send queued email to user
        try {
            Mail::to($deposit->user->email)->queue(new DepositPendingMail($deposit));
        } catch (\Exception $e) {
            Log::error('DepositPendingMail failed to send: ' . $e->getMessage());
        }

        // ✅ Send queued email to admin (from WebsiteSetting)
        try {
            $adminEmail = WebsiteSetting::value('site_email');
            if ($adminEmail) {
                Mail::to($adminEmail)->queue(new NewDepositAlertMail($deposit));
            } else {
                Log::warning('Admin email not configured in WebsiteSetting.');
            }
        } catch (\Exception $e) {
            Log::error('NewDepositAlertMail failed to send: ' . $e->getMessage());
        }

        try {
            // 🔹 Notify Admin
            if ($admin) {
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $admin->id,
                    'type' => 'New Deposit',
                    'data' => [
                        'message' => "{$user->firstname} {$user->lastname} made a deposit of {$deposit->amount}",
                        'deposit_id' => $deposit->id,
                        'status' => 'Pending',
                    ],
                ]);
            }

            // 🔹 Notify User
            Notification::create([
                'id' => Str::uuid(),
                'notifiable_type' => User::class,
                'notifiable_id' => $user->id,
                'type' => 'Deposit Submitted',
                'data' => [
                    'message' => "Your deposit of {$deposit->amount} has been received and is pending admin approval.",
                    'deposit_id' => $deposit->id,
                    'status' => 'Pending',
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Deposit notifications failed: ' . $e->getMessage());
        }

        return redirect()
            ->route('user.statement')
            ->with('success', 'Deposit submitted successfully!');
    }


    //ADMIN SECTION
    /**
     * Display a listing of the resource.
     */


    public function admindepositindex(Request $request)
    {
        $pageTitle = 'Deposits';

        $deposit = Deposit::orderBy('created_at', 'desc')->paginate(25);

        return view('admin/deposit/index', compact('pageTitle', 'deposit'));
    }

    public function admindepositdetails(Request $request, $id)
    {
        $deposit = Deposit::findorfail($id);

        $pageTitle = 'Deposits Details';

        return view('admin/deposit/details', compact('pageTitle', 'deposit'));
    }

    public function admindepositstatus(Request $request, $id)
    {
        $deposit = Deposit::findOrFail($id);

        $validated = $request->validate([
            'status' => 'required|in:pending,approved,rejected',
        ]);

        $newStatus = $validated['status'];
        $oldStatus = $deposit->status;
        $user = $deposit->user; // assumes Deposit belongsTo User

        // ✅ Only proceed if there's a change in status
        if ($oldStatus === $newStatus) {
            return back()->with('info', 'No status change detected.');
        }

        // ✅ Handle balance adjustment logic
        if ($oldStatus !== 'approved' && $newStatus === 'approved') {
            // Approving: credit user balance
            $user->balance += $deposit->amount;
            $user->save();
        } elseif ($oldStatus === 'approved' && $newStatus !== 'approved') {
            // Reverting approval: deduct user balance if enough funds exist
            if ($user->balance >= $deposit->amount) {
                $user->balance -= $deposit->amount;
                $user->save();
            } else {
                return back()->with('error', 'Cannot revert approval — user balance too low.');
            }
        }

        // ✅ Update deposit status
        $deposit->status = $newStatus;
        $deposit->save();

        // ✅ Notify user based on new status
        try {
            if ($newStatus === 'approved') {
                Mail::to($user->email)->queue(new DepositApprovedMail($deposit));
            } elseif ($newStatus === 'rejected') {
                Mail::to($user->email)->queue(new DepositRejectedMail($deposit));
            }
        } catch (\Exception $e) {
            Log::error('Deposit status mail failed: ' . $e->getMessage());
        }

        try {
            $admin = User::where('role', 'admin')->first();

            // 🔹 User notification
            $userMessage = match ($newStatus) {
                'approved' => "Your deposit of {$deposit->amount} has been approved and credited to your account.",
                'rejected' => "Your deposit of {$deposit->amount} was rejected. Please review and re-submit.",
                default => "Your deposit status was updated to {$newStatus}.",
            };

            Notification::create([
                'id' => Str::uuid(),
                'notifiable_type' => User::class,
                'notifiable_id' => $user->id,
                'type' => "Deposit {$newStatus}",
                'data' => [
                    'message' => $userMessage,
                    'deposit_id' => $deposit->id,
                    'status' => $newStatus,
                ],
            ]);

            // 🔹 Admin notification (log of status change)
            if ($admin) {
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $admin->id,
                    'type' => "Deposit {$newStatus} Update",
                    'data' => [
                        'message' => "You updated {$user->firstname} {$user->lastname}'s deposit (₦{$deposit->amount}) to {$newStatus}.",
                        'deposit_id' => $deposit->id,
                        'status' => $newStatus,
                    ],
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Deposit notification creation failed: ' . $e->getMessage());
        }

        return redirect()->route('admin.deposit.index')->with('success', 'Deposit status updated successfully.');
    }


    public function adminpaymentgateway()
    {
        $pageTitle = 'Payment Gateways';

        $payment = PaymentGateway::all();

        return view('admin/deposit/gateway', compact('pageTitle', 'payment'));
    }
    public function adminpaymentgatewayadd()
    {
        $pageTitle = 'Add Payment Gateways';

        return view('admin/deposit/add_gateway', compact('pageTitle'));
    }

    public function adminpaymentgatewayedit($id)
    {
        $payment = PaymentGateway::findOrFail($id);

        $pageTitle = 'Manage ' . $payment->name . ' Gateway';

        return view('admin.deposit.edit_gateway', compact('pageTitle', 'payment'));
    }

    public function adminpaymentgatewaystore(Request $request)
    {
        // dd($request->all());
        $payment = new PaymentGateway;

        $payment->name = $request->name;
        $payment->currency = $request->currency;
        $payment->network = $request->network;
        $payment->wallet_address = $request->wallet_address;
        $payment->min_amount = $request->min_amount;
        $payment->instructions = $request->instructions;
        $payment->status = $request->status;

        if (!empty($request->file('logo'))) {

            $file = $request->file('logo');
            $randomStr = Str::random(30);
            $filename = $randomStr . '.' . $file->getClientOriginalExtension();
            $file->move('upload/', $filename);
            $payment->logo = $filename;
        }

        if (!empty($request->file('qr_code'))) {

            $file = $request->file('qr_code');
            $randomStr = Str::random(30);
            $filename = $randomStr . '.' . $file->getClientOriginalExtension();
            $file->move('upload/', $filename);
            $payment->qr_code = $filename;
        }

        $payment->save();

        return redirect()->route('admin.gateway.index')->with('success', 'Payment Gateway Added Successfully');
    }

    public function adminpaymentgatewayupdate(Request $request, $id)
    {
        // dd($request->all());
        $payment = PaymentGateway::findorfail($id);

        $payment->name = $request->name;
        $payment->currency = $request->currency;
        $payment->network = $request->network;
        $payment->wallet_address = $request->wallet_address;
        $payment->min_amount = $request->min_amount;
        $payment->instructions = $request->instructions;
        $payment->status = $request->status;

        if (!empty($request->file('logo'))) {

            $file = $request->file('logo');
            $randomStr = Str::random(30);
            $filename = $randomStr . '.' . $file->getClientOriginalExtension();
            $file->move('upload/', $filename);
            $payment->logo = $filename;
        }

        if (!empty($request->file('qr_code'))) {

            $file = $request->file('qr_code');
            $randomStr = Str::random(30);
            $filename = $randomStr . '.' . $file->getClientOriginalExtension();
            $file->move('upload/', $filename);
            $payment->qr_code = $filename;
        }

        $payment->save();

        return redirect()->route('admin.gateway.index')->with('success', 'Payment Gateway Updated Successfully');
    }

    public function adminpaymentdestroy(Request $request, $id)
    {
        $request = Paymentgateway::find($id);
        $request->delete();
        return redirect()->route('admin.payment.method')->with('message', 'Payment Gateway Deleted Successfully');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(Deposit $deposit)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Deposit $deposit)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Deposit $deposit)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Deposit $deposit)
    {
        //
    }
}
