<?php

namespace App\Http\Controllers\Auth;

use App\Models\User;
use App\Mail\WelcomeMail;
use Illuminate\Support\Facades\Log;
use App\Mail\VerificationCodeMail;
use Illuminate\Support\Facades\Mail;
use App\Models\WebsiteSetting;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Models\Country;
use App\Models\Currency;
use Illuminate\View\View;
use Illuminate\Http\Request;
use Illuminate\Validation\Rules;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\Session;
use Illuminate\Auth\Events\Registered;

class RegisteredUserController extends Controller
{
    /**
     * Display the registration view.
     */
    public function create(): View
    {
        $country = Country::where('status', '1')
            ->orderBy('name', 'asc')
            ->get();

        $currency = Currency::where('status', '1')->get();

        $pageTitle = 'Register';

        $captcha = $this->generateCaptcha(6);

        session(['captcha_code' => $captcha]);

        return view('auth.register', compact('country', 'pageTitle', 'currency', 'captcha'));
    }

    protected function generateCaptcha(int $length = 6): string
    {
        $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
        $code = '';
        for ($i = 0; $i < $length; $i++) {
            $code .= $chars[random_int(0, strlen($chars) - 1)];
        }
        return $code;
    }

    public function refreshCaptcha(Request $request)
    {
        $code = $this->generateCaptcha(6);
        $request->session()->put('captcha_code', $code);
        // return JSON so JS can update UI
        return response()->json(['captcha' => $code]);
    }

    /**
     * Handle an incoming registration request.
     *
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(Request $request): RedirectResponse
    {
        $request->validate([
            'firstname' => ['required', 'string', 'max:255', 'regex:/^[^<>]*$/'],
            'lastname' => ['required', 'string', 'max:255', 'regex:/^[^<>]*$/'],
            'username' => ['required', 'string', 'max:255', 'regex:/^[^<>]*$/', 'unique:' . User::class],
            'email' => ['required', 'string', 'lowercase', 'email', 'max:255', 'unique:' . User::class],
            'account_type' => ['required', 'string', 'max:255'],
            'country' => ['required', 'string', 'max:255'],
            'phone' => ['required', 'string', 'max:255', 'regex:/^[^<>]*$/'],
            'referral_code' => ['nullable', 'string', 'max:255', 'regex:/^[^<>]*$/'],
            'currency' => ['required', 'string', 'max:255'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'captcha' => ['required', function ($attribute, $value, $fail) use ($request) {
                $sessionCode = $request->session()->get('captcha_code');
                if (!$sessionCode || strtoupper(trim($value)) !== $sessionCode) {
                    $fail('Invalid captcha code.');
                }
            }],
        ]);

        DB::beginTransaction();

        try {
            $newReferralCode = strtoupper(Str::random(8));
            $verificationCode = random_int(100000, 999999);

            $user = User::create([
                'firstname' => $request->firstname,
                'lastname'  => $request->lastname,
                'username'  => $request->username,
                'email'     => $request->email,
                'account_type' => $request->account_type,
                'country'   => $request->country,
                'phone'     => $request->phone,
                'currency'  => $request->currency,
                'password'  => Hash::make($request->password),
                'referral_code' => $newReferralCode,
                'status'    => 'Active',
                'verification_code' => $verificationCode,
            ]);

            logUserActivity('Registration', 'New user registered');

            $user->update([
                'last_login_at' => now(),
                'last_seen_at'  => now(),
            ]);

            // ✅ Referral bonus
            if ($request->filled('referral_code')) {
                $referrer = User::where('referral_code', $request->referral_code)->first();
                if ($referrer) {
                    $refBonus = WebsiteSetting::value('ref_bonus') ?? 0;
                    $referrer->increment('bonus', $refBonus);

                    DB::table('referrals')->insert([
                        'referrer_id' => $referrer->id,
                        'referred_id' => $user->id,
                        'bonus' => $refBonus,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                    Log::info('Referral bonus applied', ['referrer' => $referrer->id]);
                }
            }

            DB::commit();

            // 🔧 Fetch site setting
            $requireVerification = WebsiteSetting::value('require_email_verification') ?? true;

            // ✅ Handle email verification toggle
            if ($requireVerification) {
                try {
                    Mail::to($user->email)->queue(new VerificationCodeMail($user));
                } catch (\Exception $mailEx) {
                    Log::error('Email queue failed', ['error' => $mailEx->getMessage()]);
                }

                $request->session()->forget('captcha_code');

                return redirect()->route('verify.email.page')
                    ->with('email', $user->email)
                    ->with('status', 'A verification code has been sent to your email.');
            } else {
                // 🚀 Skip verification
                $user->update(['email_verified_at' => now()]);
                Auth::login($user);

                return redirect()->route('user.dashboard')
                    ->with('success', 'Registration successful! Email verification skipped as per system setting.');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->withErrors(['error' => 'Registration failed: ' . $e->getMessage()]);
        }
    }

    public function verifyEmailPage()
    {
        return view('auth.verify-email');
    }

    public function verifyEmailSubmit(Request $request)
    {
        $request->validate([
            'email' => 'required|email|exists:users,email',
            'code' => 'required|string|max:6',
        ]);

        $user = User::where('email', $request->email)
            ->where('verification_code', $request->code)
            ->first();

        if (!$user) {
            return back()->withErrors(['code' => 'Invalid verification code.']);
        }

        $user->update([
            'email_verified_at' => now(),
            'verification_code' => null,
        ]);

        Auth::login($user);

        // ✅ Send welcome email (queued)
        try {
            Mail::to($user->email)->queue(new WelcomeMail($user));
        } catch (\Exception $e) {
            Log::error('Welcome email failed', ['error' => $e->getMessage()]);
        }

        return redirect()->route('user.dashboard')
            ->with('success', 'Email verified successfully!.');
    }

    public function resendVerification(Request $request)
    {
        // ✅ Get email from session or request
        $email = $request->email ?? session('email') ?? session('unverified_email');

        if (!$email) {
            return redirect()->route('login')
                ->with('error', 'Session expired. Please log in again to resend verification.');
        }

        // ✅ Find user by email
        $user = User::where('email', $email)->first();

        if (!$user) {
            return redirect()->route('login')->with('error', 'User not found.');
        }

        // ✅ If already verified
        if ($user->email_verified_at) {
            return redirect()->route('user.dashboard')->with('message', 'Your email is already verified.');
        }

        // ✅ Generate a new verification code
        $newCode = rand(100000, 999999);
        $user->update(['verification_code' => $newCode]);

        try {
            Mail::to($user->email)->queue(new VerificationCodeMail($user));
        } catch (\Exception $e) {
            return back()->with('error', 'Failed to resend verification code. Please try again.');
        }

        return back()->with('message', 'A new verification code has been sent to your email.');
    }
}
