<?php

namespace App\Http\Controllers;

use App\Models\AiBot;
use App\Mail\BotTradeStoppedMail;
use App\Models\WebsiteSetting;
use App\Mail\BotTradeStartedMail;
use App\Mail\NewBotTradeAlertMail;
use Illuminate\Support\Facades\Mail;
use App\Models\User;
use Illuminate\Support\Str;
use App\Models\Notification;
use Illuminate\Support\Facades\Log;
use App\Models\UserBotTrade;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;


class AiBotController extends Controller
{
    // Show single bot details
    public function userbotdetails($name)
    {
        $bot = AiBot::whereRaw('LOWER(name) = ?', [strtolower($name)])->firstOrFail();

        $user = Auth::user();

        $pageTitle = 'Invest on ' . $bot->name . ' Bot';

        $userTrade = UserBotTrade::where('user_id', $user->id)
            ->where('ai_bot_id', $bot->id)
            ->latest()
            ->first();

        return view('user.bot_details', compact('bot', 'pageTitle', 'userTrade'));
    }


    // Start trading with a bot
    public function startbottrade(Request $request, AiBot $bot)
    {
        if (is_null($bot->min_amount) || is_null($bot->max_amount)) {
            return back()->withErrors(['error' => 'Bot configuration is incomplete. Please contact admin.']);
        }

        $request->validate([
            'amount' => [
                'required',
                'numeric',
                'min:' . ($bot->min_amount ?? 0),
                'max:' . ($bot->max_amount ?? 9999999),
            ],
        ]);

        $user = Auth::user();
        $amount = $request->amount;

        // ✅ Check if user has enough balance
        if ($user->balance < $amount) {
            return back()->withErrors(['error' => 'Insufficient balance to start this trade.']);
        }

        // ✅ Wrap logic in transaction to keep things consistent
        DB::transaction(function () use ($bot, $user, $amount) {

            // Deduct the invested amount from user balance
            $user->decrement('balance', $amount);

            // Calculate trade end date
            $endsAt = now()->addDays($bot->duration_days);

            // Create the trade record
            $trade = UserBotTrade::create([
                'user_id'         => $user->id,
                'ai_bot_id'       => $bot->id,
                'amount_invested' => $amount,
                'current_value'   => $amount,
                'profit_earned'   => 0,
                'profit_percent'  => 0,
                'started_at'      => now(),
                'ends_at'         => $endsAt,
                'status'          => 'running',
            ]);

            // Increment bot’s active users
            $bot->increment('active_users');

            // 🔔 Notify Admin & User
            $admin = User::where('role', 'admin')->first();
            $adminEmail = WebsiteSetting::value('site_email');

            try {
                // ✉️ Send email to user
                Mail::to($user->email)->send(new BotTradeStartedMail($trade));

                // ✉️ Send email to admin
                if ($adminEmail) {
                    Mail::to($adminEmail)->send(new NewBotTradeAlertMail($trade));
                }
            } catch (\Exception $e) {
                Log::error('Bot trade email failed: ' . $e->getMessage());
            }


            try {
                // 🔹 Admin notification
                if ($admin) {
                    Notification::create([
                        'id' => Str::uuid(),
                        'notifiable_type' => User::class,
                        'notifiable_id' => $admin->id,
                        'type' => 'New Bot Trade',
                        'data' => [
                            'message' => "{$user->firstname} {$user->lastname} started a {$bot->name} trade with $$amount",
                            'bot_id' => $bot->id,
                            'status' => 'Running',
                        ],
                    ]);
                }

                // 🔹 User notification
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $user->id,
                    'type' => 'Bot Trade Started',
                    'data' => [
                        'message' => "You successfully started a {$bot->name} trade with $$amount. Duration: {$bot->duration_days} days.",
                        'bot_id' => $bot->id,
                        'status' => 'Running',
                    ],
                ]);
            } catch (\Exception $e) {
            }

            // 🧾 Log activity
            logUserActivity('AI Bot Trade', "Started {$bot->name} bot trade with $$amount for {$bot->duration_days} days");
        });

        return back()->with('success', 'Trading started successfully with ' . $bot->name);
    }


    // Stop trading manually
    public function stopBotTradeByBotName(AiBot $bot)
    {
        $user = Auth::user();

        // 🧠 Find user's active trade with this bot
        $trade = UserBotTrade::where('user_id', $user->id)
            ->where('ai_bot_id', $bot->id)
            ->where('status', 'running')
            ->first();

        if (!$trade) {
            return back()->with('error', 'No active trade found for this bot.');
        }

        // ✅ Only stop if trade has value
        if ($trade->current_value > 0) {
            DB::transaction(function () use ($user, $trade, $bot) {

                // Credit current value to user
                $user->increment('profit', $trade->current_value);

                // Mark trade as completed
                $trade->update(['status' => 'completed']);

                // Log final daily record
                DB::table('bot_trade_daily_records')->insert([
                    'user_bot_trade_id' => $trade->id,
                    'percent' => $trade->profit_percent,
                    'profit' => $trade->profit_earned,
                    'recorded_at' => now()->toDateString(),
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                // 🧾 Activity Log
                logUserActivity('AI Bot Trade', "Stopped {$bot->name} bot trade and credited $" . number_format($trade->current_value, 2));

                // 🔔 Notification
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $user->id,
                    'type' => 'Bot Trade Stopped',
                    'data' => [
                        'message' => "You stopped your {$bot->name} trade and received $" . number_format($trade->current_value, 2),
                        'status' => 'Completed',
                    ],
                ]);

                // 📧 Email (optional, if you have BotTradeStoppedMail)
                try {
                    Mail::to($user->email)->send(new BotTradeStoppedMail($trade));
                } catch (\Exception $e) {
                    Log::error("BotTradeStoppedMail failed: " . $e->getMessage());
                }
            });

            return back()->with('success', "Trade stopped successfully. Your returns have been credited to your profit.");
        }

        return back()->with('error', 'Trade value too low or already closed.');
    }


    public function dailyBotProfitUpdate()
    {
        // load running trades + bot
        $runningTrades = UserBotTrade::with('bot')
            ->where('status', 'running')
            ->get();

        foreach ($runningTrades as $trade) {
            $bot = $trade->bot;

            // --- parse min/max from bot->daily_profit like "0.80% - 2.50%"
            preg_match('/([\d.]+)\s*%\s*-\s*([\d.]+)\s*%/', $bot->daily_profit, $matches);
            if (count($matches) === 3) {
                $minProfit = (float)$matches[1]; // e.g. 0.8
                $maxProfit = (float)$matches[2]; // e.g. 2.5
            } else {
                // default fallback
                $minProfit = 0.8;
                $maxProfit = 2.5;
            }

            // ensure sensible ordering
            if ($minProfit > $maxProfit) {
                [$minProfit, $maxProfit] = [$maxProfit, $minProfit];
            }

            // determine win vs loss using success_rate (0-100)
            $successRate = (float)$bot->success_rate; // ensure saved as numeric (e.g. 87.54)
            $isWin = (mt_rand(1, 10000) / 100) <= max(0, min(100, $successRate));

            // choose random percent within range
            $randomCents = mt_rand((int)($minProfit * 100), (int)($maxProfit * 100)); // integer cents
            $dailyPercent = $randomCents / 100.0; // e.g. 1.37

            // if loss, negate percent (optionally scale losses)
            if (! $isWin) {
                // you can tune loss severity here. using same range but negative:
                $dailyPercent = -1 * $dailyPercent;
            }

            // compute profit (signed)
            $profitToday = $trade->amount_invested * ($dailyPercent / 100);

            // prevent over-penalizing (e.g. more than invested)
            if ($profitToday <= -1 * $trade->amount_invested) {
                $profitToday = -1 * $trade->amount_invested * 0.999; // never more than 99.9% loss
            }

            // begin transaction to keep data consistent
            DB::transaction(function () use ($trade, $bot, $dailyPercent, $profitToday) {
                // increment numeric columns (make sure columns are numeric in DB)
                $trade->increment('profit_earned', $profitToday);
                // profit_percent accumulates percent values (store total percent, not daily)
                $trade->increment('profit_percent', abs($dailyPercent)); // keep track of cumulative percent magnitude
                // create a daily record for auditing
                DB::table('bot_trade_daily_records')->insert([
                    'user_bot_trade_id' => $trade->id,
                    'percent' => $dailyPercent,
                    'profit' => $profitToday,
                    'recorded_at' => now()->toDateString(),
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                // update bot totals and handle completion
                $bot->increment('total_earned', $profitToday);

                // if trade has reached or passed ends_at, complete it
                if (now()->gte($trade->ends_at)) {
                    $trade->update(['status' => 'completed']);
                    // decrement active users but guard against negative:
                    if ($bot->active_users > 0) {
                        $bot->decrement('active_users');
                    }
                }
            });

            // log what happened (helpful)
            Log::info("Bot daily update: trade_id={$trade->id} bot_id={$bot->id} percent={$dailyPercent} profit={$profitToday}");
        }

        Log::info('Daily bot profit update completed at ' . now());
    }


    // public function dailyBotProfitUpdate()
    // {
    //     $runningTrades = UserBotTrade::with('bot')
    //         ->where('status', 'running')
    //         ->get();

    //     $updated = 0;

    //     foreach ($runningTrades as $trade) {
    //         $bot = $trade->bot;

    //         // Parse min and max profit percentages from the bot’s daily_profit string
    //         preg_match('/([\d.]+)\s*%\s*-\s*([\d.]+)\s*%/', $bot->daily_profit, $matches);

    //         if (count($matches) === 3) {
    //             $minProfit = (float)$matches[1];
    //             $maxProfit = (float)$matches[2];
    //         } else {
    //             $minProfit = 0.8;
    //             $maxProfit = 2.5;
    //         }

    //         // Random daily profit within range
    //         $dailyProfit = mt_rand($minProfit * 100, $maxProfit * 100) / 100;

    //         // Calculate today's profit
    //         $profitToday = $trade->amount_invested * ($dailyProfit / 100);

    //         // Update trade incrementally
    //         $trade->increment('profit_earned', $profitToday);
    //         $trade->increment('profit_percent', $dailyProfit);

    //         // Check for completion
    //         if (now()->gte($trade->ends_at)) {
    //             $trade->update(['status' => 'completed']);
    //         }

    //         // Update bot’s total earned
    //         $bot->increment('total_earned', $profitToday);

    //         $updated++;
    //     }

    //     return response()->json([
    //         'message' => '✅ Daily bot profit update completed.',
    //         'updated_trades' => $updated,
    //         'timestamp' => now()->toDateTimeString()
    //     ]);
    // }


    //ADMIN BOT SECTION
    public function adminaibotindex(Request $request)
    {
        $pageTitle = 'Ai Bots';

        $bots = AiBot::all();

        return view('admin/aibot/index', compact('pageTitle', 'bots'));
    }

    public function adminaibotcreate(Request $request)
    {
        $pageTitle = 'Add New Bots';

        return view('admin/aibot/create', compact('pageTitle'));
    }

    public function aibotstore(Request $request)
    {

        $request->validate([
            'name' => 'required|string|max:255',
            'success_rate' => 'nullable|numeric|min:0|max:100',
            'trading_type' => 'required|string|max:255',
            'daily_profit' => 'nullable|string',
            'duration_days' => 'nullable|integer|min:0',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'nullable|numeric|min:0',
            'trading_frequency' => 'nullable|string|max:255',
            'strategy_type' => 'nullable|string|max:255',
            'trading_pairs' => 'nullable|string|max:255',
            'active_users' => 'nullable|integer|min:0',
            'total_earned' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'status' => 'nullable|string',
            'expected_returns' => 'nullable|string',
            'total_trades' => 'nullable|integer|min:0',
            'daily_change' => 'nullable|numeric',
            'bots_profit' => 'nullable|numeric',
        ]);

        $bot = new AiBot;
        $bot->name = $request->name;
        $bot->trading_type = $request->trading_type;
        $bot->success_rate = $request->success_rate;
        $bot->daily_profit = $request->daily_profit;
        $bot->duration_days = $request->duration_days;
        $bot->min_amount = $request->min_amount;
        $bot->max_amount = $request->max_amount;
        $bot->trading_frequency = $request->trading_frequency;
        $bot->strategy_type = $request->strategy_type;
        $bot->trading_pairs = $request->trading_pairs;
        $bot->active_users = $request->active_users;
        $bot->total_earned = $request->total_earned;
        $bot->description = $request->description;
        $bot->expected_returns = $request->expected_returns;
        $bot->bots_profit = $request->bots_profit;
        $bot->total_trades = $request->total_trades;
        $bot->daily_change = $request->daily_change;
        $bot->status = $request->status;

        $bot->save();

        return redirect()->route('admin.aibot.index')
            ->with('success', 'AiBot created successfully!');
    }

    public function adminaibotedit(Request $request, $id)
    {
        $bot = AiBot::findOrFail($id);

        $pageTitle = 'Update ' . $bot->name . ' Details';

        return view('admin/aibot/edit', compact('bot', 'pageTitle'));
    }

    public function aibotupdate(Request $request, $id)
    {

        $bot = AiBot::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'success_rate' => 'nullable|numeric|min:0|max:100',
            'trading_type' => 'required|string|max:255',
            'daily_profit' => 'nullable|string',
            'duration_days' => 'nullable|integer|min:0',
            'min_amount' => 'required|numeric|min:0',
            'max_amount' => 'nullable|numeric|min:0',
            'trading_frequency' => 'nullable|string|max:255',
            'strategy_type' => 'nullable|string|max:255',
            'trading_pairs' => 'nullable|string|max:255',
            'active_users' => 'nullable|integer|min:0',
            'total_earned' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'expected_returns' => 'nullable|string',
            'total_trades' => 'nullable|integer|min:0',
            'daily_change' => 'nullable|numeric',
        ]);

        $bot->name = $request->name;
        $bot->trading_type = $request->trading_type;
        $bot->success_rate = $request->success_rate;
        $bot->daily_profit = $request->daily_profit;
        $bot->duration_days = $request->duration_days;
        $bot->min_amount = $request->min_amount;
        $bot->max_amount = $request->max_amount;
        $bot->trading_frequency = $request->trading_frequency;
        $bot->strategy_type = $request->strategy_type;
        $bot->trading_pairs = $request->trading_pairs;
        $bot->active_users = $request->active_users;
        $bot->total_earned = $request->total_earned;
        $bot->description = $request->description;
        $bot->expected_returns = $request->expected_returns;
        $bot->total_trades = $request->total_trades;
        $bot->daily_change = $request->daily_change;

        $bot->save();

        return redirect()->route('admin.aibot.index')
            ->with('success', 'AiBot updated successfully!');
    }

    public function destroy($id)
    {
        $bot = AiBot::findOrFail($id);
        $bot->delete();

        return redirect()->route('admin.aibots.index')
            ->with('success', 'AiBot deleted successfully!');
    }


    public function adminaibottrades(Request $request)
    {
        $bot = UserBotTrade::latest()->paginate(25);

        $pageTitle = 'User Bot Trades';

        return view('admin/aibot/bot_trades', compact('bot', 'pageTitle'));
    }
}
