<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Notification;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use App\Mail\InvestmentCompletedMail;
use App\Models\Investment;
use App\Models\InvestmentLog;
use Carbon\Carbon;

class UpdateInvestments extends Command
{
    protected $signature = 'investments:update';
    protected $description = 'Update investment profits daily, and credit users only at maturity';

    public function handle()
    {
        $investments = Investment::where('status', 'active')->get();
        $this->info("Found {$investments->count()} active investments...");
        $now = Carbon::now();
        $today = $now->toDateString();

        foreach ($investments as $investment) {
            try {
                $plan = $investment->plan;
                if (!$plan) {
                    $this->warn("Skipping investment #{$investment->id} (no plan)");
                    continue;
                }

                $startDate = Carbon::parse($investment->start_date);
                $endDate   = Carbon::parse($investment->end_date);

                // Previous profit BEFORE any write
                $previousProfit = (float) $investment->profit;

                // How many cycles have passed
                $cycles = $this->getCompletedCycles($plan, $startDate, $now);

                // Profit per cycle and total accumulated profit (rounded)
                $profitPerCycle = round(($investment->amount * $plan->interest_rate) / 100, 2);
                $totalProfit    = round($profitPerCycle * $cycles, 2);

                // Cap total profit to expected_return - amount (the contract profit)
                $maxProfit = round($investment->expected_return - $investment->amount, 2);
                if ($totalProfit > $maxProfit) {
                    $totalProfit = $maxProfit;
                }
                if ($totalProfit < 0) {
                    $totalProfit = 0;
                }

                // How much more profit since last run
                $profitGained = round($totalProfit - $previousProfit, 2);

                // --- Create daily log (profit_progress) if there is new profit AND not logged today ---
                if ($profitGained > 0) {
                    $alreadyLogged = InvestmentLog::where('investment_id', $investment->id)
                        ->whereDate('created_at', $today)
                        ->where('type', 'profit_progress')
                        ->exists();

                    if (!$alreadyLogged) {
                        InvestmentLog::create([
                            'user_id'       => $investment->user_id,
                            'investment_id' => $investment->id,
                            'amount'        => $profitGained,
                            'type'          => 'profit_progress',
                            'note'          => "Profit progress logged on {$today} (not credited)",
                        ]);
                        $this->info("Logged profit_progress for investment #{$investment->id}: +{$profitGained}");
                    } else {
                        $this->info("Profit already logged today for investment #{$investment->id}");
                    }
                } else {
                    $this->info("No new profit for investment #{$investment->id} today (gained={$profitGained})");
                }

                // --- Update investment row (persist totalProfit and days_completed) ---
                // Use forceFill to avoid fillable issues; you can use mass assignment if your model's fillable is configured.
                $investment->forceFill([
                    'profit' => $totalProfit,
                    'days_completed' => $startDate->diffInDays($now),
                ])->save();

                $this->info("Updated investment #{$investment->id}: profit={$totalProfit}, days_completed={$investment->days_completed}");

                // --- If matured, finalize: set status, create completed log if absent and credit once ---
                if ($now->greaterThanOrEqualTo($endDate)) {
                    // ensure final values
                    $investment->forceFill([
                        'status' => 'completed',
                        'profit' => $maxProfit,
                        'days_completed' => $this->getTotalDays($plan),
                    ])->save();

                    $alreadyCompleted = InvestmentLog::where('investment_id', $investment->id)
                        ->where('type', 'completed')
                        ->exists();

                    if (!$alreadyCompleted) {
                        InvestmentLog::create([
                            'user_id'       => $investment->user_id,
                            'investment_id' => $investment->id,
                            'amount'        => $maxProfit,
                            'type'          => 'completed',
                            'note'          => "Investment completed and profit credited on {$today}",
                        ]);

                        // Credit user once with the total profit (not principal)
                        $user = $investment->user;

                        if ($user) {

                            // ---------------------------------------
                            // 1️⃣ CREDIT USER PROFIT
                            // ---------------------------------------
                            $user->increment('profit', $maxProfit);

                            // ---------------------------------------
                            // 2️⃣ SEND USER EMAIL NOTIFICATION
                            // ---------------------------------------
                            try {
                                Mail::to($user->email)->queue(new InvestmentCompletedMail($investment));
                            } catch (\Exception $e) {
                                Log::error("InvestmentCompletedMail failed: " . $e->getMessage());
                            }

                            // ---------------------------------------
                            // 3️⃣ CREATE USER IN-APP NOTIFICATION
                            // ---------------------------------------
                            try {
                                Notification::create([
                                    'id' => \Illuminate\Support\Str::uuid(),
                                    'notifiable_type' => \App\Models\User::class,
                                    'notifiable_id' => $user->id,
                                    'type' => 'Investment Completed',
                                    'data' => [
                                        'message' => "Your investment in {$investment->plan->name} has been completed. Profit credited: $" . number_format($maxProfit, 2),
                                        'investment_id' => $investment->id,
                                        'status' => 'Completed'
                                    ],
                                ]);
                            } catch (\Exception $e) {
                                Log::error("Investment completed notification failed: " . $e->getMessage());
                            }

                            // ---------------------------------------
                            // 4️⃣ LOG ACTIVITY
                            // ---------------------------------------
                            logUserActivity(
                                'Investment Completed',
                                "Completed investment #{$investment->id} ({$investment->plan->name}) Profit: $" . number_format($maxProfit, 2)
                            );

                            $this->info("🏁 Investment #{$investment->id} completed — notifications + email sent.");
                        } else {
                            $this->warn("No user found to credit for investment #{$investment->id}");
                        }
                    } else {
                        $this->info("Completion already logged for investment #{$investment->id}");
                    }
                }
            } catch (\Throwable $e) {
                $this->error("Error processing investment #{$investment->id}: " . $e->getMessage());
            }
        }


        $this->info('✅ Daily profit updated (user credit only on completion) at ' . now());
    }

    /**
     * Calculate total plan duration in days
     */
    private function getTotalDays($plan)
    {
        return match (strtolower($plan->interest_type)) {
            'daily'   => $plan->duration,
            'weekly'  => $plan->duration * 7,
            'monthly' => $plan->duration * 30,
            'yearly'  => $plan->duration * 365,
            default   => $plan->duration,
        };
    }

    /**
     * Determine how many cycles have elapsed since start date
     */
    private function getCompletedCycles($plan, $startDate, $now)
    {
        return match (strtolower($plan->interest_type)) {
            'daily'   => $startDate->diffInDays($now),
            'weekly'  => floor($startDate->diffInWeeks($now)),
            'monthly' => floor($startDate->diffInMonths($now)),
            'yearly'  => floor($startDate->diffInYears($now)),
            default   => 0,
        };
    }
}
