<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use App\Models\User;
use App\Models\Staking;
use Illuminate\Support\Str;
use App\Models\Notification;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use App\Mail\UserStakingCompletedMail;

class ProcessStakingMaturity extends Command
{
    protected $signature = 'staking:process';
    protected $description = 'Recalculate interest and credit matured stakings';

    public function handle()
    {
        $now = Carbon::now();

        $stakings = Staking::where('status', 'active')
            ->whereNull('completed_at')
            ->where('end_date', '<=', $now)
            ->get();

        if ($stakings->isEmpty()) {
            $this->info('No matured stakings.');
            return;
        }

        foreach ($stakings as $stake) {

            $user = null;
            $totalReturn = 0;

            DB::transaction(function () use ($stake, &$user, &$totalReturn) {

                $user = User::find($stake->user_id);
                if (!$user) return;

                $amount = $stake->amount;
                $rate   = $stake->interest_rate;
                $days   = $stake->duration;

                $interest = round(($amount * $rate / 100) * $days, 2);
                $totalReturn = $amount + $interest;

                $stake->interest_earned = $interest;
                $stake->status = 'completed';
                $stake->completed_at = now();
                $stake->save();

                $user->increment('profit', $totalReturn);
            });

            if ($user) {
                try {

                    // EMAIL
                    Mail::to($user->email)
                        ->queue(new UserStakingCompletedMail($user, $stake, $totalReturn));

                    // NOTIFICATION
                    Notification::create([
                        'id' => Str::uuid(),
                        'notifiable_type' => User::class,
                        'notifiable_id' => $user->id,
                        'type' => 'Staking Completed',
                        'data' => [
                            'message' => 'Your staking has completed and wallet credited.',
                            'plan' => $stake->stakingPlan->name ?? '',
                            'amount' => $stake->amount,
                            'interest' => $stake->interest_earned,
                            'total_paid' => $totalReturn,
                            'status' => 'Completed',
                        ],
                    ]);
                } catch (\Exception $e) {
                    Log::error($e->getMessage());
                }
            }

            $this->info("Stake {$stake->id} completed & user notified.");
        }


        $this->info('All matured stakes processed.');
    }
}
