<?php

namespace App\Console\Commands;

use Carbon\Carbon;
use App\Models\User;
use App\Models\LiveTrade;
use Illuminate\Support\Str;
use App\Models\Notification;
use App\Mail\TradeClosedMail;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;

class ProcessLiveTrades extends Command
{
    protected $signature = 'trades:process';
    protected $description = 'Update live trades using Binance and TwelveData, close expired ones, and compute P/L.';

    public function handle()
    {
        $this->info('🔄 Processing open live trades...');
        $trades = LiveTrade::where('status', 'open')->get();

        foreach ($trades as $trade) {
            $this->info("Checking trade #{$trade->id} ({$trade->symbol})...");

            // ✅ Auto-close expired trades
            if ($trade->expires_at && Carbon::parse($trade->expires_at)->isPast()) {
                $this->closeTrade($trade, 'completed');
                continue;
            }

            // ✅ Fetch live price
            $currentPrice = $this->fetchCurrentPrice($trade->symbol, $trade->market_type);

            if (!$currentPrice) {
                $this->warn("⚠️ Could not fetch price for {$trade->symbol}");
                continue;
            }

            // ✅ Calculate current value and P/L
            $trade->current_value = $this->calculateCurrentValue($trade, $currentPrice);
            $trade->profit_loss = $trade->current_value - $trade->amount;

            // 🚨 Check for liquidation (loss exceeds investment)
            if ($trade->profit_loss <= -$trade->amount) {
                $this->warn("💥 Trade #{$trade->id} liquidated (loss exceeded margin).");

                // Mark trade as closed, user loses full investment
                $trade->current_value = 0;
                $trade->profit_loss = -$trade->amount;
                $this->closeTrade($trade, 'closed');
                continue;
            }

            $trade->save();

            $this->info("✅ Trade #{$trade->id} updated | Price: {$currentPrice} | P/L: {$trade->profit_loss}");
        }

        $this->info('🎯 All live trades processed successfully.');
    }

    /**
     * Fetch current price using Binance (crypto) or TwelveData (stocks/forex)
     */
    private function fetchCurrentPrice($symbol, $marketType)
    {
        // ✅ Binance for crypto
        if ($marketType === 'crypto' || str_ends_with($symbol, 'USDT')) {
            $response = Http::get("https://api.binance.com/api/v3/ticker/price", [
                'symbol' => strtoupper($symbol),
            ])->json();

            if (isset($response['code'])) {
                Log::warning("❌ Binance error for {$symbol}: " . json_encode($response));
                return null;
            }

            return isset($response['price']) ? (float) $response['price'] : null;
        }

        // ✅ TwelveData for stocks/forex
        $response = Http::get("https://api.twelvedata.com/quote", [
            'symbol' => $symbol,
            'apikey' => config('services.twelvedata.key'),
        ])->json();

        if (isset($response['close'])) {
            return (float) $response['close'];
        }

        Log::warning("❌ TwelveData error for {$symbol}: " . json_encode($response));
        return null;
    }

    /**
     * Calculate the current value of the trade
     */
    private function calculateCurrentValue($trade, $currentPrice)
    {
        $priceChange = $currentPrice - $trade->entry_price;

        // For sell trades, profit/loss direction is reversed
        if ($trade->side === 'sell') {
            $priceChange = -$priceChange;
        }

        // Leverage-adjusted profit/loss
        $profitLoss = ($priceChange / $trade->entry_price) * $trade->amount * $trade->leverage;
        return $trade->amount + $profitLoss;
    }

    /**
     * Close trade automatically or manually
     */
    private function closeTrade(LiveTrade $trade, $reason = 'completed')
    {
        if ($trade->status !== 'open') {
            Log::info("⚠️ Trade #{$trade->id} already {$trade->status}, skipping credit.");
            return;
        }

        $user = $trade->user;
        $creditedAmount = 0;

        // CREDIT
        if ($user && $trade->current_value > 0) {
            $creditedAmount = $trade->current_value;
            $user->profit += $creditedAmount;
            $user->save();

            Log::info("💰 Credited user #{$user->id} with {$creditedAmount} for trade #{$trade->id}");
        } else {
            Log::info("💥 Trade #{$trade->id} liquidated — no credit given.");
        }

        // CLOSE TRADE
        $trade->update([
            'status' => $reason,
            'expires_at' => now(),
        ]);

        Log::info("✅ Trade #{$trade->id} marked as {$reason}");

        // NOTIFY USER
        if ($user) {
            try {

                // EMAIL
                Mail::to($user->email)
                    ->queue(new TradeClosedMail($user, $trade, $reason));

                // IN-APP NOTIFICATION
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $user->id,
                    'type' => 'Trade Closed',
                    'data' => [
                        'message' => "Your trade #{$trade->id} has been {$reason}.",
                        'trade_id' => $trade->id,
                        'credited_amount' => $creditedAmount,
                        'status' => ucfirst($reason),
                    ],
                ]);
            } catch (\Exception $e) {
                Log::error("Trade close notify failed: " . $e->getMessage());
            }
        }
    }

    // private function closeTrade(LiveTrade $trade, $reason = 'completed')
    // {
    //     // Avoid double crediting
    //     if ($trade->status !== 'open') {
    //         Log::info("⚠️ Trade #{$trade->id} already {$trade->status}, skipping credit.");
    //         return;
    //     }

    //     $user = $trade->user;

    //     // 💰 Only credit if trade still has value
    //     if ($user && $trade->current_value > 0) {
    //         $user->profit += $trade->current_value;
    //         $user->save();

    //         Log::info("💰 Credited user #{$user->id} with {$trade->current_value} for trade #{$trade->id}");
    //     } else {
    //         Log::info("💥 Trade #{$trade->id} liquidated — no credit given.");
    //     }

    //     // 🏁 Close trade
    //     $trade->update([
    //         'status' => $reason,
    //         'expires_at' => now(),
    //     ]);

    //     Log::info("✅ Trade #{$trade->id} marked as {$reason}");
    // }
}
